#include "Block.h"

#include <QDebug>

const double L = 10.0;

LongBlock::LongBlock()
{
	Build(0.0);
}
void LongBlock::Build(double alpha)
{
	A(-sin(alpha)-cos(alpha), 0.0, -cos(alpha)+sin(alpha));
	B(-sin(alpha)+cos(alpha), 0.0, -cos(alpha)-sin(alpha));
	C(sin(alpha)+cos(alpha), 0.0, cos(alpha)-sin(alpha));
	D(sin(alpha)-cos(alpha), 0.0, cos(alpha)+sin(alpha));
	
	a = Project(A, YY, XX * L, (XX + YY).Normalised());
	b = Project(B, YY, XX * L, (XX + YY).Normalised());
	c = Project(C, YY, XX * L, (XX + YY).Normalised());
	d = Project(D, YY, XX * L, (XX + YY).Normalised());

}
void LongBlock::Render()
{
	glBegin(GL_QUADS);
	glQuad(A, B, C, D);
	glQuad(b, a, d, c);
	glQuad(a, b, B, A);
	glQuad(b, c, C, B);
	glQuad(c, d, D, C);
	glQuad(d, a, A, D);
	glEnd();
}

LongBlock LongBlock::Reflected(const Vector& Offset, const Vector& Normal)
{
	LongBlock L;
	
	// Have to reverse directions to make it not inside out.
	L.A = ReflectOffset(D, Offset, Normal);
	L.B = ReflectOffset(C, Offset, Normal);
	L.C = ReflectOffset(B, Offset, Normal);
	L.D = ReflectOffset(A, Offset, Normal);
	L.a = ReflectOffset(d, Offset, Normal);
	L.b = ReflectOffset(c, Offset, Normal);
	L.c = ReflectOffset(b, Offset, Normal);
	L.d = ReflectOffset(a, Offset, Normal);
	
	return L;
}

void LongBlock::RenderHinges()
{	
	glPushAttrib(GL_CURRENT_BIT);
	glPushAttrib(GL_ENABLE_BIT);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glColor4d(0.0, 1.0, 0.0, 1.0);
	glBegin(GL_LINES);
	glVertex(a); glVertex(b);
	glEnd();
	glPopAttrib();
	glPopAttrib();
	
}

Vector LongBlock::Centre()
{
	return (A + B + C + D + a + b + c + d)/8.0;
}
	
ShortBlock::ShortBlock()
{
	Build(0.0, 0.0, 0.0);
}
	
void ShortBlock::Build(double alpha, double beta, double psi)
{
	// First get the long end geometry.
	Vector A(-sin(alpha)-cos(alpha), 0.0, -cos(alpha)+sin(alpha));
	Vector B(-sin(alpha)+cos(alpha), 0.0, -cos(alpha)-sin(alpha));
	Vector C(sin(alpha)+cos(alpha), 0.0, cos(alpha)-sin(alpha));
	Vector D(sin(alpha)-cos(alpha), 0.0, cos(alpha)+sin(alpha));
	
	Vector a = Project(A, YY, XX * L, (XX + YY).Normalised());
	Vector b = Project(B, YY, XX * L, (XX + YY).Normalised());
	Vector c = Project(C, YY, XX * L, (XX + YY).Normalised());
	Vector d = Project(D, YY, XX * L, (XX + YY).Normalised());
	
	// Now reflect a b c d in the C D c d plane.
	
	g = c;
	h = d;
	e = ReflectOffset(a, C, ((c - C) ^ (D - C)).Normalised());
	f = ReflectOffset(b, C, ((c - C) ^ (D - C)).Normalised());
	
	// And for the other end.
	Vector Gtmp = C;
	Vector Htmp = D;
	Vector Etmp = ReflectOffset(A, C, ((c - C) ^ (D - C)).Normalised());
	Vector Ftmp = ReflectOffset(B, C, ((c - C) ^ (D - C)).Normalised());
	
	// Now we need to rotate EFGH by beta - alpha anticlockwise looking in the Y direction.
	E = Rotate(Etmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta - alpha) * -YY);
	F = Rotate(Ftmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta - alpha) * -YY);
	G = Rotate(Gtmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta - alpha) * -YY);
	H = Rotate(Htmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta - alpha) * -YY);
	
	double dd = beta - alpha;
	while (dd < 0.0) dd += 2.0 * Pi;
	while (dd > 2.0 * Pi) dd -= 2.0 * Pi;
	if (dd < Pi/2.0)
		rotation = 0;
	else if (dd < Pi)
		rotation = 1;
	else if (dd < 3.0 * Pi / 2.0)
		rotation = 2;
	else
		rotation = 3;
	
	// Now create the middle twist by projecting efgh and EFGH onto some intermediate planes.
	
	et = ProjectOntoPlaneOffset(e, 3.0*L/4.0 * YY, YY);
	ft = ProjectOntoPlaneOffset(f, 3.0*L/4.0 * YY, YY);
	gt = ProjectOntoPlaneOffset(g, 3.0*L/4.0 * YY, YY);
	ht = ProjectOntoPlaneOffset(h, 3.0*L/4.0 * YY, YY);
	Et = ProjectOntoPlaneOffset(E, L/4.0 * YY, YY);
	Ft = ProjectOntoPlaneOffset(F, L/4.0 * YY, YY);
	Gt = ProjectOntoPlaneOffset(G, L/4.0 * YY, YY);
	Ht = ProjectOntoPlaneOffset(H, L/4.0 * YY, YY);
	
	// Now rotate the whole lot by psi about the hinge.
	
	e = Rotate(e, c, (d-c).Normalised() * psi);
	f = Rotate(f, c, (d-c).Normalised() * psi);
	g = Rotate(g, c, (d-c).Normalised() * psi);
	h = Rotate(h, c, (d-c).Normalised() * psi);

	E = Rotate(E, c, (d-c).Normalised() * psi);
	F = Rotate(F, c, (d-c).Normalised() * psi);
	G = Rotate(G, c, (d-c).Normalised() * psi);
	H = Rotate(H, c, (d-c).Normalised() * psi);

	et = Rotate(et, c, (d-c).Normalised() * psi);
	ft = Rotate(ft, c, (d-c).Normalised() * psi);
	gt = Rotate(gt, c, (d-c).Normalised() * psi);
	ht = Rotate(ht, c, (d-c).Normalised() * psi);

	Et = Rotate(Et, c, (d-c).Normalised() * psi);
	Ft = Rotate(Ft, c, (d-c).Normalised() * psi);
	Gt = Rotate(Gt, c, (d-c).Normalised() * psi);
	Ht = Rotate(Ht, c, (d-c).Normalised() * psi);
}
	
Vector ShortBlock::GetPlaneOffset()
{
	return E;
}
Vector ShortBlock::GetPlaneNormal()
{
	Vector a = Reflect(H, YY);
	return ((E-H)^(H-a)).Normalised();
}
	
void ShortBlock::Render()
{
	glBegin(GL_QUADS);
	
	// Ends
	glQuad(e, f, g, h);
	glQuad(H, G, F, E);
	
	// Sides
	
	// First segment.
	
	glQuad(et, ft, f, e);
	glQuad(ft, gt, g, f);
	glQuad(gt, ht, h, g);
	glQuad(ht, et, e, h);
	
	// Last segment.
	
	glQuad(E, F, Ft, Et);
	glQuad(F, G, Gt, Ft);
	glQuad(G, H, Ht, Gt);
	glQuad(H, E, Et, Ht);
	
	glEnd();
	glBegin(GL_TRIANGLES);
	
	// Middle segment - use triangles.
	
	switch (rotation)
	{
	default:
	case 3:
		glTriangle(Et, ft, et); glTriangle(Et, Ft, ft);
		glTriangle(Ft, gt, ft); glTriangle(Ft, Gt, gt);
		glTriangle(Gt, ht, gt); glTriangle(Gt, Ht, ht);
		glTriangle(Ht, et, ht); glTriangle(Ht, Et, et);
		break;
	case 0:
		glTriangle(Ft, ft, et); glTriangle(Ft, Gt, ft);
		glTriangle(Gt, gt, ft); glTriangle(Gt, Ht, gt);
		glTriangle(Ht, ht, gt); glTriangle(Ht, Et, ht);
		glTriangle(Et, et, ht); glTriangle(Et, Ft, et);
		break;
	case 1:
		glTriangle(Gt, ft, et); glTriangle(Gt, Ht, ft);
		glTriangle(Ht, gt, ft); glTriangle(Ht, Et, gt);
		glTriangle(Et, ht, gt); glTriangle(Et, Ft, ht);
		glTriangle(Ft, et, ht); glTriangle(Ft, Gt, et);
		break;
	case 2:
		glTriangle(Ht, ft, et); glTriangle(Ht, Et, ft);
		glTriangle(Et, gt, ft); glTriangle(Et, Ft, gt);
		glTriangle(Ft, ht, gt); glTriangle(Ft, Gt, ht);
		glTriangle(Gt, et, ht); glTriangle(Gt, Ht, et);
		break;
	}

	glEnd();
}

void ShortBlock::RenderHinges()
{	
	glPushAttrib(GL_CURRENT_BIT);
	glPushAttrib(GL_ENABLE_BIT);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glColor4d(0.0, 1.0, 0.0, 1.0);
	glBegin(GL_LINES);
	glVertex(E); glVertex(H);
	glVertex(h); glVertex(g);
	glEnd();
	glPopAttrib();
	glPopAttrib();
	
}
	
ShortBlock ShortBlock::Reflected(const Vector& Offset, const Vector& Normal)
{
	ShortBlock S;
	
	// Have to reverse directions to make it not inside out.
	S.E = ReflectOffset(H, Offset, Normal);
	S.F = ReflectOffset(G, Offset, Normal);
	S.G = ReflectOffset(F, Offset, Normal);
	S.H = ReflectOffset(E, Offset, Normal);
	S.e = ReflectOffset(h, Offset, Normal);
	S.f = ReflectOffset(g, Offset, Normal);
	S.g = ReflectOffset(f, Offset, Normal);
	S.h = ReflectOffset(e, Offset, Normal);
	
	S.Et = ReflectOffset(Ht, Offset, Normal);
	S.Ft = ReflectOffset(Gt, Offset, Normal);
	S.Gt = ReflectOffset(Ft, Offset, Normal);
	S.Ht = ReflectOffset(Et, Offset, Normal);
	S.et = ReflectOffset(ht, Offset, Normal);
	S.ft = ReflectOffset(gt, Offset, Normal);
	S.gt = ReflectOffset(ft, Offset, Normal);
	S.ht = ReflectOffset(et, Offset, Normal);
	
	S.rotation = 3-rotation;
	
	return S;
}


