#include "Vector.h"

/* Reflect V about a plane with a normalised normal Normal. :) */
Vector Reflect(const Vector& Point, const Vector& Normal)
{
	return Point - Normal * 2 * (Point * Normal);	
}
/* Project a point onto the plane that goes through the origin with normal Normal.
   Normal must be normalised. The point is translated parallel to the Normal. */
Vector ProjectOntoPlane(const Vector& Point, const Vector& Normal)
{
	return Point - (Normal * (Normal * Point));
}
/* Get the nearest distance of a point from the line P-Q. */
double DistanceToLine(const Vector& P, const Vector& Q, const Vector& Point)
{
	Vector TmpPlaneNormal = Q - P;
	TmpPlaneNormal.Normalise();
	if (TmpPlaneNormal * (Point - Q) > 0.0) // Past Q.
	{
		return (Point - Q).Magnitude();
	}
	if (TmpPlaneNormal * (Point - P) < 0.0) // Past P.
	{
		return (Point - P).Magnitude();
	}
	return ProjectOntoPlane(Point - P, TmpPlaneNormal).Magnitude(); // Along line.
}
/* Get the distance between two vectors. */
double Distance(const Vector& P, const Vector& Q)
{
	return (P - Q).Magnitude();
}

// Reflect Point in a plane at position Offset with normal Normal.
Vector ReflectOffset(const Vector& Point, const Vector& Offset, const Vector& Normal)
{
	return Point - Normal * 2 * ((Point - Offset) * Normal);	
}

// Project Point perpendicularly onto an offset plane.
Vector ProjectOntoPlaneOffset(const Vector& Point, const Vector& Offset, const Vector& Normal)
{
	return Point - Normal * ((Point - Offset) * Normal);		
}

// Project Point in the direction Project onto an offset plane. Project must be normalised.
Vector Project(const Vector& Point, const Vector& Project, const Vector& Offset, const Vector& Normal)
{
	if (Project * Normal == 0.0)
		return Point; // Can't do this.
	
	return Point + Project * (((Offset - Point) * Normal) / (Project * Normal));
}

// Rotate a point about an axis. Offset is a point on the axis, Axis is the axis * the rotation.
Vector Rotate(const Vector& Point, const Vector& Offset, const Vector& Axis)
{
	double c = cos(Axis.Magnitude());
	double s = sin(Axis.Magnitude());
	double t = 1.0-c;
	Vector A = Axis.Normalised();
	
	Vector V = Point - Offset;
	
	return Offset + Vector(
		V.X * (t * A.X * A.X + c) + V.Y * (t * A.X * A.Y - s * A.Z) + V.Z * (t * A.X * A.Z + s * A.Y),
		V.X * (t * A.X * A.Y + s * A.Z) + V.Y * (t * A.Y * A.Y + c) + V.Z * (t * A.Y * A.Z - s * A.X),
		V.X * (t * A.X * A.Z - s * A.Y) + V.Y * (t * A.Y * A.Z + s * A.X) + V.Z * (t * A.Z * A.Z + c));
}

