#include "Block.h"

#include "Graphics.h"

#include <QDebug>

// Edge length.
const double L = 100.0;

/** ************* BLOCK 2 *****************/

Block2::Block2()
{
	Build(0.0, 0.0);
}
void Block2::Build(double alpha1, double alpha2)
{
	A(-sin(alpha1)-cos(alpha1), 0.0, -cos(alpha1)+sin(alpha1));
	B(-sin(alpha1)+cos(alpha1), 0.0, -cos(alpha1)-sin(alpha1));
	C(sin(alpha1)+cos(alpha1), 0.0, cos(alpha1)-sin(alpha1));
	D(sin(alpha1)-cos(alpha1), 0.0, cos(alpha1)+sin(alpha1));

	At = A - YY * L/4.0;
	Bt = B - YY * L/4.0;
	Ct = C - YY * L/4.0;
	Dt = D - YY * L/4.0;
	
	A = Project(A, YY, XX * L, (XX - YY).Normalised());
	B = Project(B, YY, XX * L, (XX - YY).Normalised());
	C = Project(C, YY, XX * L, (XX - YY).Normalised());
	D = Project(D, YY, XX * L, (XX - YY).Normalised());

	a(-sin(alpha2)-cos(alpha2), 0.0, -cos(alpha2)+sin(alpha2));
	b(-sin(alpha2)+cos(alpha2), 0.0, -cos(alpha2)-sin(alpha2));
	c(sin(alpha2)+cos(alpha2), 0.0, cos(alpha2)-sin(alpha2));
	d(sin(alpha2)-cos(alpha2), 0.0, cos(alpha2)+sin(alpha2));
	
	at = a + YY * L/4.0;
	bt = b + YY * L/4.0;
	ct = c + YY * L/4.0;
	dt = d + YY * L/4.0;

	a = Project(a, YY, XX * L, (XX + YY).Normalised());
	b = Project(b, YY, XX * L, (XX + YY).Normalised());
	c = Project(c, YY, XX * L, (XX + YY).Normalised());
	d = Project(d, YY, XX * L, (XX + YY).Normalised());
	
	rotation = RotationFromAngleOffset(alpha1 - alpha2);
}
void Block2::Render() const
{

}

Block2 Block2::Reflected(const Vector& Offset, const Vector& Normal) const
{
	Block2 L;
	
	L.A = ReflectOffset(A, Offset, Normal);
	L.B = ReflectOffset(B, Offset, Normal);
	L.C = ReflectOffset(C, Offset, Normal);
	L.D = ReflectOffset(D, Offset, Normal);
	L.a = ReflectOffset(a, Offset, Normal);
	L.b = ReflectOffset(b, Offset, Normal);
	L.c = ReflectOffset(c, Offset, Normal);
	L.d = ReflectOffset(d, Offset, Normal);

	L.At = ReflectOffset(At, Offset, Normal);
	L.Bt = ReflectOffset(Bt, Offset, Normal);
	L.Ct = ReflectOffset(Ct, Offset, Normal);
	L.Dt = ReflectOffset(Dt, Offset, Normal);
	L.at = ReflectOffset(at, Offset, Normal);
	L.bt = ReflectOffset(bt, Offset, Normal);
	L.ct = ReflectOffset(ct, Offset, Normal);
	L.dt = ReflectOffset(dt, Offset, Normal);

	L.rotation = 3-rotation;
	
	return L;
}

void Block2::RenderHinges() const
{	

}

Vector Block2::Centre() const
{
	return (A + B + C + D + a + b + c + d)/8.0;
}

/** ************* BLOCK 1 *****************/

Block1::Block1()
{
	Build(0.0, 0.0, 0.0, 0.0);
}

void Block1::Build(double alpha1, double beta1, double alpha2, double beta2)
{
	// First get the block2 geometry at the block1 end.
	Vector A(-sin(alpha1)-cos(alpha1), 0.0, -cos(alpha1)+sin(alpha1));
	Vector B(-sin(alpha1)+cos(alpha1), 0.0, -cos(alpha1)-sin(alpha1));
	Vector C(sin(alpha1)+cos(alpha1), 0.0, cos(alpha1)-sin(alpha1));
	Vector D(sin(alpha1)-cos(alpha1), 0.0, cos(alpha1)+sin(alpha1));

	Vector c = C;
	
	A = Project(A, YY, XX * L, (XX - YY).Normalised());
	B = Project(B, YY, XX * L, (XX - YY).Normalised());
	C = Project(C, YY, XX * L, (XX - YY).Normalised());
	D = Project(D, YY, XX * L, (XX - YY).Normalised());
	
	// Now reflect a b c d in the C D c d plane.
	
	G = C;
	H = D;
	E = ReflectOffset(A, C, ((c - C) ^ (D - C)).Normalised());
	F = ReflectOffset(B, C, ((c - C) ^ (D - C)).Normalised());
	
	// And for the other end.
	Vector etmp = ProjectOntoPlaneOffset(E, YY * L/2.0, YY);
	Vector ftmp = ProjectOntoPlaneOffset(F, YY * L/2.0, YY);
	Vector gtmp = ProjectOntoPlaneOffset(G, YY * L/2.0, YY);
	Vector htmp = ProjectOntoPlaneOffset(H, YY * L/2.0, YY);
	
	// Now we need to rotate EFGH by beta - alpha anticlockwise looking in the Y direction.
	e = Rotate(etmp, (gtmp + htmp + etmp + ftmp)/4.0, (beta1 - alpha1) * -YY);
	f = Rotate(ftmp, (gtmp + htmp + etmp + ftmp)/4.0, (beta1 - alpha1) * -YY);
	g = Rotate(gtmp, (gtmp + htmp + etmp + ftmp)/4.0, (beta1 - alpha1) * -YY);
	h = Rotate(htmp, (gtmp + htmp + etmp + ftmp)/4.0, (beta1 - alpha1) * -YY);

	rotation = RotationFromAngleOffset(beta1 - alpha1);
	
	// Now create the middle twist by projecting efgh and EFGH onto some intermediate planes.
	
	et = ProjectOntoPlaneOffset(e, L/4.0 * YY, YY);
	ft = ProjectOntoPlaneOffset(f, L/4.0 * YY, YY);
	gt = ProjectOntoPlaneOffset(g, L/4.0 * YY, YY);
	ht = ProjectOntoPlaneOffset(h, L/4.0 * YY, YY);
	Et = ProjectOntoPlaneOffset(E, -L/4.0 * YY, YY);
	Ft = ProjectOntoPlaneOffset(F, -L/4.0 * YY, YY);
	Gt = ProjectOntoPlaneOffset(G, -L/4.0 * YY, YY);
	Ht = ProjectOntoPlaneOffset(H, -L/4.0 * YY, YY);
	
	// Now we must (possibly) shift the hinge a bit depending on which is further out.
	// Use the other alpha, beta's to work out where the other hinge is.
	
	Vector thisHinge(2.0 * sin(alpha1) - cos(2.0 * alpha1 - beta1), 0.0, 2.0 * cos(alpha1) + sin(2.0 * alpha1 - beta1));
	Vector otherHinge(2.0 * sin(alpha2) - cos(2.0 * alpha2 - beta2), 0.0, 2.0 * cos(alpha2) + sin(2.0 * alpha2 - beta2));
	
	// Should be same as alpha2, beta2.
	Vector direction(-cos(2.0 * alpha1 - beta1), 0.0, sin(2.0 * alpha1 - beta1));
	
	double offSet = otherHinge * direction - thisHinge * direction;
	
	if (offSet > 0.0)
	{
		// Shift h and e.
		
		h += direction * offSet;
		e += direction * offSet;
	}
}

void Block1::Open(double psi12)
{
	// Rotate the whole lot by psi about the hinge.
	Vector GG = G;
	Vector HH = H;
	
	e = Rotate(e, GG, (GG-HH).Normalised() * psi12);
	f = Rotate(f, GG, (GG-HH).Normalised() * psi12);
	g = Rotate(g, GG, (GG-HH).Normalised() * psi12);
	h = Rotate(h, GG, (GG-HH).Normalised() * psi12);

	E = Rotate(E, GG, (GG-HH).Normalised() * psi12);
	F = Rotate(F, GG, (GG-HH).Normalised() * psi12);
	G = Rotate(G, GG, (GG-HH).Normalised() * psi12);
	H = Rotate(H, GG, (GG-HH).Normalised() * psi12);

	et = Rotate(et, GG, (GG-HH).Normalised() * psi12);
	ft = Rotate(ft, GG, (GG-HH).Normalised() * psi12);
	gt = Rotate(gt, GG, (GG-HH).Normalised() * psi12);
	ht = Rotate(ht, GG, (GG-HH).Normalised() * psi12);

	Et = Rotate(Et, GG, (GG-HH).Normalised() * psi12);
	Ft = Rotate(Ft, GG, (GG-HH).Normalised() * psi12);
	Gt = Rotate(Gt, GG, (GG-HH).Normalised() * psi12);
	Ht = Rotate(Ht, GG, (GG-HH).Normalised() * psi12);
}

void Block1::Render() const
{

}

void Block1::RenderHinges() const
{	

}
	
Block1 Block1::Reflected(const Vector& Offset, const Vector& Normal) const
{
	Block1 S;

	// Don't reverse here otherwise the hinge offsets don't work and we don't
	// care about rendering anyway.
	S.E = ReflectOffset(E, Offset, Normal);
	S.F = ReflectOffset(F, Offset, Normal);
	S.G = ReflectOffset(G, Offset, Normal);
	S.H = ReflectOffset(H, Offset, Normal);
	S.e = ReflectOffset(e, Offset, Normal);
	S.f = ReflectOffset(f, Offset, Normal);
	S.g = ReflectOffset(g, Offset, Normal);
	S.h = ReflectOffset(h, Offset, Normal);
	
	S.Et = ReflectOffset(Et, Offset, Normal);
	S.Ft = ReflectOffset(Ft, Offset, Normal);
	S.Gt = ReflectOffset(Gt, Offset, Normal);
	S.Ht = ReflectOffset(Ht, Offset, Normal);
	S.et = ReflectOffset(et, Offset, Normal);
	S.ft = ReflectOffset(ft, Offset, Normal);
	S.gt = ReflectOffset(gt, Offset, Normal);
	S.ht = ReflectOffset(ht, Offset, Normal);
	
	S.rotation = 3-rotation;
	
	return S;
}

Vector Block1::GetSymmetryHingeOffset() const
{
	return h;
}
Vector Block1::GetSymmetryHingeDirection() const
{
	return (e - h).Normalised();
}
Vector Block1::GetFoldingHingeOffset() const
{
	return G;
}
Vector Block1::GetFoldingHingeDirection() const
{
	return (G - H).Normalised();
}



/** ************* BLOCK 3 *****************/

Block3::Block3()
{
	Build(0.0, 0.0, 0.0, 0.0);
}
	
void Block3::Build(double alpha1, double beta1, double alpha2, double beta2)
{
	// First get the block2 geometry at the block3 end.
	Vector a(-sin(alpha2)-cos(alpha2), 0.0, -cos(alpha2)+sin(alpha2));
	Vector b(-sin(alpha2)+cos(alpha2), 0.0, -cos(alpha2)-sin(alpha2));
	Vector c(sin(alpha2)+cos(alpha2), 0.0, cos(alpha2)-sin(alpha2));
	Vector d(sin(alpha2)-cos(alpha2), 0.0, cos(alpha2)+sin(alpha2));

	Vector C = c; // For the reflection plane.
	Vector D = d;
	
	a = Project(a, YY, XX * L, (XX + YY).Normalised());
	b = Project(b, YY, XX * L, (XX + YY).Normalised());
	c = Project(c, YY, XX * L, (XX + YY).Normalised());
	d = Project(d, YY, XX * L, (XX + YY).Normalised());
	
	// Now reflect a b c d in the C D c d plane.
	
	g = c;
	h = d;
	e = ReflectOffset(a, C, ((c - C) ^ (D - C)).Normalised());
	f = ReflectOffset(b, C, ((c - C) ^ (D - C)).Normalised());
	
	// And for the other end.
	Vector Etmp = ProjectOntoPlaneOffset(e, YY * -L/2.0, YY);
	Vector Ftmp = ProjectOntoPlaneOffset(f, YY * -L/2.0, YY);
	Vector Gtmp = ProjectOntoPlaneOffset(g, YY * -L/2.0, YY);
	Vector Htmp = ProjectOntoPlaneOffset(h, YY * -L/2.0, YY);
	
	// Now we need to rotate EFGH by beta - alpha anticlockwise looking in the Y direction.
	E = Rotate(Etmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta2 - alpha2) * -YY);
	F = Rotate(Ftmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta2 - alpha2) * -YY);
	G = Rotate(Gtmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta2 - alpha2) * -YY);
	H = Rotate(Htmp, (Gtmp + Htmp + Etmp + Ftmp)/4.0, (beta2 - alpha2) * -YY);

	rotation = RotationFromAngleOffset(beta2 - alpha2);
	
	// Now create the middle twist by projecting efgh and EFGH onto some intermediate planes.
	
	et = ProjectOntoPlaneOffset(e, L/4.0 * YY, YY);
	ft = ProjectOntoPlaneOffset(f, L/4.0 * YY, YY);
	gt = ProjectOntoPlaneOffset(g, L/4.0 * YY, YY);
	ht = ProjectOntoPlaneOffset(h, L/4.0 * YY, YY);
	Et = ProjectOntoPlaneOffset(E, -L/4.0 * YY, YY);
	Ft = ProjectOntoPlaneOffset(F, -L/4.0 * YY, YY);
	Gt = ProjectOntoPlaneOffset(G, -L/4.0 * YY, YY);
	Ht = ProjectOntoPlaneOffset(H, -L/4.0 * YY, YY);
	
	// Now we must (possibly) shift the hinge a bit depending on which is further out.
	// Use the other alpha, beta's to work out where the other hinge is.

	Vector thisHinge(2.0 * sin(alpha2) - cos(2.0 * alpha2 - beta2), 0.0, 2.0 * cos(alpha2) + sin(2.0 * alpha2 - beta2));
	Vector otherHinge(2.0 * sin(alpha1) - cos(2.0 * alpha1 - beta1), 0.0, 2.0 * cos(alpha1) + sin(2.0 * alpha1 - beta1));
	
	// Should be same as alpha1, beta1.
	Vector direction(-cos(2.0 * alpha2 - beta2), 0.0, sin(2.0 * alpha2 - beta2));
	
	double offSet = otherHinge * direction - thisHinge * direction;
	
	if (offSet > 0.0)
	{
		// Shift H and E.
		
		H += direction * offSet;
		E += direction * offSet;
	}	
}

void Block3::Open(double psi23)
{
	// Rotate the whole lot by psi about the hinge.

	Vector gg = g;
	Vector hh = h;
	
	e = Rotate(e, gg, (hh-gg).Normalised() * psi23);
	f = Rotate(f, gg, (hh-gg).Normalised() * psi23);
	g = Rotate(g, gg, (hh-gg).Normalised() * psi23);
	h = Rotate(h, gg, (hh-gg).Normalised() * psi23);

	E = Rotate(E, gg, (hh-gg).Normalised() * psi23);
	F = Rotate(F, gg, (hh-gg).Normalised() * psi23);
	G = Rotate(G, gg, (hh-gg).Normalised() * psi23);
	H = Rotate(H, gg, (hh-gg).Normalised() * psi23);

	et = Rotate(et, gg, (hh-gg).Normalised() * psi23);
	ft = Rotate(ft, gg, (hh-gg).Normalised() * psi23);
	gt = Rotate(gt, gg, (hh-gg).Normalised() * psi23);
	ht = Rotate(ht, gg, (hh-gg).Normalised() * psi23);

	Et = Rotate(Et, gg, (hh-gg).Normalised() * psi23);
	Ft = Rotate(Ft, gg, (hh-gg).Normalised() * psi23);
	Gt = Rotate(Gt, gg, (hh-gg).Normalised() * psi23);
	Ht = Rotate(Ht, gg, (hh-gg).Normalised() * psi23);
}

void Block3::Render() const
{

}

void Block3::RenderHinges() const
{	
}
	
Block3 Block3::Reflected(const Vector& Offset, const Vector& Normal) const
{
	Block3 S;
	
	// Have to reverse directions to make it not inside out.
	S.E = ReflectOffset(E, Offset, Normal);
	S.F = ReflectOffset(F, Offset, Normal);
	S.G = ReflectOffset(G, Offset, Normal);
	S.H = ReflectOffset(H, Offset, Normal);
	S.e = ReflectOffset(e, Offset, Normal);
	S.f = ReflectOffset(f, Offset, Normal);
	S.g = ReflectOffset(g, Offset, Normal);
	S.h = ReflectOffset(h, Offset, Normal);
	
	S.Et = ReflectOffset(Et, Offset, Normal);
	S.Ft = ReflectOffset(Ft, Offset, Normal);
	S.Gt = ReflectOffset(Gt, Offset, Normal);
	S.Ht = ReflectOffset(Ht, Offset, Normal);
	S.et = ReflectOffset(et, Offset, Normal);
	S.ft = ReflectOffset(ft, Offset, Normal);
	S.gt = ReflectOffset(gt, Offset, Normal);
	S.ht = ReflectOffset(ht, Offset, Normal);
	
	S.rotation = 3-rotation;
	
	return S;
}


Vector Block3::GetSymmetryHingeOffset() const
{
	return E;
}
Vector Block3::GetSymmetryHingeDirection() const
{
	return (H - E).Normalised();
}
Vector Block3::GetFoldingHingeOffset() const
{
	return g;
}
Vector Block3::GetFoldingHingeDirection() const
{
	return (h - g).Normalised();
}
