

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "MainWindow.h"

#include <QMessageBox>

#include <QImage>
#include "Block.h"

#include "Maths.h"

#include <QDebug>




bool MechanismWorks(double alpha1, double alpha2, double gamma)
{
	
	if (alpha1 < -Pi)
		alpha1 = -Pi;
	if (alpha1 > Pi)
		alpha1 = Pi;
	if (alpha2 < -Pi)
		alpha2 = -Pi;
	if (alpha2 > Pi)
		alpha2 = Pi;
	if (gamma < -Pi)
		gamma = -Pi;
	if (gamma > Pi)
		gamma = Pi;

	for (double psi = 0.001; psi <= PsiMax(alpha2); psi += Pi2/300.0)
	{
		Block1 block1;
		Block2 block2;
		Block3 block3;
		
		block1.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);
		block2.Build(alpha1, alpha2);
		block3.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);
	
		block3.Open(psi);
	
		// Calculate the two psi12 values.
	
		// Make the maths not pathlogocial.
		// TODO: If F or f == 0 (or less than some small value), then special case.
		pair<double, double> psi12s =
			RotationsForPlane(block3.GetSymmetryHingeOffset(), block3.GetSymmetryHingeDirection(),
							block1.GetSymmetryHingeOffset(), block1.GetSymmetryHingeDirection(),
							block1.GetFoldingHingeOffset(), block1.GetFoldingHingeDirection());
		
		double sln = psi12s.second;
		
		// Avoid crazy solutions.
		if (sln < 1000.0 && sln > -1000.0)
		{
			//			secondWorks = false;
			block1.Open(sln);
		}
		
		if ((sln < 0.0 || sln > PsiMax(alpha1)))
			return false;
		
		if (psi < Pi2/300.0 + 0.01 && abs(sln) > PsiMax(alpha1) / 10.0)
			return false;
		
		if (psi > PsiMax(alpha2) - Pi2/300.0 - 0.01 && abs(sln - PsiMax(alpha1)) > PsiMax(alpha1) / 10.0)
			return false;

		double error = abs((block1.GetSymmetryHingeDirection() ^ (block3.GetSymmetryHingeOffset() - block1.GetSymmetryHingeOffset()))
	                   * (block3.GetSymmetryHingeOffset() + block3.GetSymmetryHingeDirection() - block1.GetSymmetryHingeOffset()));
		
		if (error > 0.001)
		{
			return false;
		}
	}
	
	return true;
}

double MechanismMinSVD(double alpha1, double alpha2, double gamma)
{
	if (alpha1 < -Pi)
		alpha1 = -Pi;
	if (alpha1 > Pi)
		alpha1 = Pi;
	if (alpha2 < -Pi)
		alpha2 = -Pi;
	if (alpha2 > Pi)
		alpha2 = Pi;
	if (gamma < -Pi)
		gamma = -Pi;
	if (gamma > Pi)
		gamma = Pi;
	
	double min = INFINITY;

	for (double psi = 0.001; psi <= PsiMax(alpha2); psi += Pi2/300.0)
	{
		Block1 block1;
		Block2 block2;
		Block3 block3;
		
		block1.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);
		block2.Build(alpha1, alpha2);
		block3.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);
	
		block3.Open(psi);
	
		// Calculate the two psi12 values.
	
		// Make the maths not pathlogocial.
		// TODO: If F or f == 0 (or less than some small value), then special case.
		pair<double, double> psi12s =
			RotationsForPlane(block3.GetSymmetryHingeOffset(), block3.GetSymmetryHingeDirection(),
							block1.GetSymmetryHingeOffset(), block1.GetSymmetryHingeDirection(),
							block1.GetFoldingHingeOffset(), block1.GetFoldingHingeDirection());
		
		double sln = psi12s.second;
		
		// Avoid crazy solutions.
		if (sln < 1000.0 && sln > -1000.0)
		{
			//			secondWorks = false;
			block1.Open(sln);
		}
		
		if ((sln < 0.0 || sln > PsiMax(alpha1)))
			return -1.0;
		
		if (psi < Pi2/300.0 + 0.01 && abs(sln) > PsiMax(alpha1) / 10.0)
			return -1.0;
		
		if (psi > PsiMax(alpha2) - Pi2/300.0 - 0.01 && abs(sln - PsiMax(alpha1)) > PsiMax(alpha1) / 10.0)
			return -1.0;
		
		double error = abs((block1.GetSymmetryHingeDirection() ^ (block3.GetSymmetryHingeOffset() - block1.GetSymmetryHingeOffset()))
	                   * (block3.GetSymmetryHingeOffset() + block3.GetSymmetryHingeDirection() - block1.GetSymmetryHingeOffset()));
		
		if (error > 0.001)
		{
			return -1.0;
		}
		
		/** Calculate the SVD. */
		
		double svd = GetSVD(block1, block3)[4];
		
		if (svd < min)
			min = svd;
	}
	
	return min;
}

int main(int argc, char *argv[])
{
	QApplication app(argc, argv);
	
	if (argc == 4)
	{
		// Generate image of gamma layer.
		
		int size = QString(argv[2]).toInt();
		if (size < 2)
			size = 2;
		
		double gamma = QString(argv[1]).toInt() * Pi2/size - Pi;
		
		if (gamma < -Pi)
			gamma = -Pi;
		if (gamma > Pi)
			gamma = Pi;
		
		QString outputFile = argv[3];
		
		QImage image(size, size, QImage::Format_ARGB32);
		
		for (int x = 0; x < size; ++x)
		{
			qDebug() << x;
			for (int y = 0; y < size; ++y)
			{
				double svd = MechanismMinSVD((Pi2 * x / size) - Pi, -((Pi2 * y / size) - Pi), gamma);
				
				if (svd >= 0.0)
					image.setPixel(x, y, qRgba(svd * 255, svd * 255, svd * 255, 255));
				else
					image.setPixel(x, y, qRgba(200, 200, 255, 255));
			}
		}
		
		if (!image.save(outputFile))
			qDebug("Error saving output.");
		
		return 0;
	}
	
	MainWindow win;

	win.showMaximized();
	return app.exec();
}
