#include "Graphics.h"
#include "ViewWidget.h"
#include "Maths.h"

#include <QMouseEvent>
#include <QDebug>

ViewWidget::ViewWidget(QWidget* parent) : QGLWidget(parent)
{
	viewTheta = 45.0;
	viewPhi = 45.0;
	viewDistance = 250.0;
	
	viewCentre.X = 15.0;
	viewCentre.Y = 0.0;
	
	alpha1 = 0.0;
	alpha2 = 0.0;
	gamma = 0.0;
	psi = 0.001;

	reBuildBlocks();
	
	whichSolution = false;
}

void ViewWidget::reBuildBlocks()
{
	block1.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);
	block2.Build(alpha1, alpha2);
	block3.Build(alpha1, 2.0 * alpha1 - gamma, alpha2, 2.0 * alpha2 - gamma);

	block3.Open(PsiMax(alpha2) * psi);

	// Calculate the two psi12 values.

	// Make the maths not pathlogocial.

	// TODO: If F or f == 0 (or less than some small value), then special case.
	pair<double, double> psi12s =
		RotationsForPlane(block3.GetSymmetryHingeOffset(), block3.GetSymmetryHingeDirection(),
		                  block1.GetSymmetryHingeOffset(), block1.GetSymmetryHingeDirection(),
		                  block1.GetFoldingHingeOffset(), block1.GetFoldingHingeDirection());
	
	double sln = whichSolution ? psi12s.first : psi12s.second;
	if (sln < 1000.0 && sln > -1000.0)
		block1.Open(sln);
}

ViewWidget::~ViewWidget()
{
}
QSize ViewWidget::minimumSizeHint() const
{
	return QSize(50, 50);
}

QSize ViewWidget::sizeHint() const
{
	return QSize(500, 500);
}

void ViewWidget::setAlpha1(double angle)
{
	alpha1 = angle;
	reBuildBlocks();
	updateGL();
}

void ViewWidget::setAlpha2(double angle)
{
	alpha2 = angle;
	reBuildBlocks();
	updateGL();
}
void ViewWidget::setGamma(double angle)
{
	gamma = angle;
	reBuildBlocks();
	updateGL();
}
void ViewWidget::setPsi(double amount)
{
	psi = amount;
	if (psi == 0.0) // Avoid evilness.
		psi = 0.001;
	if (psi == 1.0)
		psi = 0.999;
	reBuildBlocks();
	updateGL();
}

#define glClearColour glClearColor

void ViewWidget::initializeGL()
{
	// Background colour.
	glClearColour(1.0, 1.0, 1.0, 0.0);
	// Transparency.
	glEnable(GL_BLEND);
	// Depth testing.
	glEnable(GL_DEPTH_TEST);
	// Don't show the back of polys.
	glEnable(GL_CULL_FACE);
	// Smoothing.
	glEnable(GL_POLYGON_SMOOTH);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	
	// Enable lighting and normal normalization.
	glEnable(GL_LIGHTING);
	glEnable(GL_NORMALIZE);
	
	// Ambient lighting.
	GLfloat ambient[4] = {0.5, 0.5, 0.5, 1.0};
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);
	
	// Light 0.
	glEnable(GL_LIGHT0);	
	GLfloat diffuse0[] = { 0.8, 0.8, 0.8, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse0);
	// Light 1.
	glEnable(GL_LIGHT1);	
	GLfloat diffuse1[] = { 0.8, 0.8, 0.8, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse1);
	
	// Colouring
	glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);
	
	// Fog
	//	glEnable(GL_FOG); When drawing the grid.
	float  fog[] = {1.0f, 1.0f, 1.0f, 0.0f};
	glFogf(GL_FOG_START,50.0f);
	glFogf(GL_FOG_END, 100.0f);
	glFogfv(GL_FOG_COLOR, fog);      
	glFogi(GL_FOG_MODE, GL_EXP);
	glFogf(GL_FOG_DENSITY, 0.04f);

}

void ViewWidget::paintGL()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	
	glTranslated(0.0, 0.0, -viewDistance/10.0);
	glRotated(viewTheta, -1.0, 0.0, 0.0);
	glRotated(viewPhi, 0.0, 0.0, 1.0);
	glTranslated(-viewCentre.X, -viewCentre.Y, 0.0);
	
	GLfloat pos0[] = { 20.0, -15.0, 10.0, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_POSITION, pos0);
	GLfloat pos1[] = { -5.0, -15.0, 10.0, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_POSITION, pos1);
	
	Vector planeOffset = block1.GetSymmetryHingeOffset();
	Vector planeNormal = (block1.GetSymmetryHingeDirection() ^ block3.GetSymmetryHingeDirection())
		+ ((block1.GetSymmetryHingeOffset() - block3.GetSymmetryHingeOffset()) ^ block1.GetSymmetryHingeDirection());
	planeNormal.Normalise();
	
	double error = abs((block1.GetSymmetryHingeDirection() ^ (block3.GetSymmetryHingeOffset() - block1.GetSymmetryHingeOffset()))
	                   * (block3.GetSymmetryHingeOffset() + block3.GetSymmetryHingeDirection() - block1.GetSymmetryHingeOffset()));
	
	if (error < 0.001)
	{
	
		glColor3d(1.0, 0.0,0.0);
		glBegin(GL_LINES);
		glVertex(planeOffset);
		glVertex(planeOffset + planeNormal);
		glEnd();
		
		Block1 block4 = block1.Reflected(planeOffset, planeNormal);
		Block2 block5 = block2.Reflected(planeOffset, planeNormal);
		Block3 block6 = block3.Reflected(planeOffset, planeNormal);
		
		glColor4d(0.5, 0.5, 0.5, 1.0);
	
		block4.Render();
		block5.Render();
		block6.Render();
	}
	block1.Render();
	block2.Render();
	block3.Render();
	block1.RenderHinges();
	block2.RenderHinges();
	block3.RenderHinges();

	glEnable(GL_FOG);
	glBegin(GL_LINES);
	glColor4d(0.5, 0.5, 0.5, 1.0);
	for (double i = -100.0; i <= 100.0; i += 10.0)
	{
		glVertex3d(-100.0, i, -10.0);
		glVertex3d(100.0, i, -10.0);
		glVertex3d(i, -100.0, -10.0);
		glVertex3d(i, 100.0, -10.0);
	}
	glEnd();
	glDisable(GL_FOG);

}

void ViewWidget::resizeGL(int width, int height)
{
	glViewport(0, 0, width, height);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (width < height)
		glFrustum(-0.1, 0.1, -0.1 * height / width, 0.1 * height / width, 0.1, 200.0);
	else
		glFrustum(-0.1 * width / height, 0.1 * width / height, -0.1, 0.1, 0.1, 200.0);
	glMatrixMode(GL_MODELVIEW);
}

void ViewWidget::mousePressEvent(QMouseEvent *event)
{
	lastPos = event->pos();
	if (event->buttons() & Qt::MidButton)
	{
		whichSolution = !whichSolution;
		reBuildBlocks();
		updateGL();
	}
}

void ViewWidget::mouseMoveEvent(QMouseEvent *event)
{
	int dx = event->x() - lastPos.x();
	int dy = event->y() - lastPos.y();
	
	if (event->buttons() & Qt::LeftButton)
	{
		viewPhi += dx;
		viewTheta -= dy;
	}
	if (event->buttons() & Qt::RightButton)
	{
		viewDistance += dy;
	}
	updateGL();
	lastPos = event->pos();
}

