#pragma once

#include <cmath>

const double Pi = 3.141592654;
const double Pi_2 = Pi/2.0;
const double Pi_3 = Pi/3.0;
const double Pi_4 = Pi/4.0;
 
const double Pi2 = 2.0*Pi;
const double Pi2_2 = 2.0*Pi/2.0;
const double Pi2_3 = 2.0*Pi/3.0;
const double Pi2_4 = 2.0*Pi/4.0;

const double Pi3 = 3.0*Pi;
const double Pi3_2 = 3.0*Pi/2.0;
const double Pi3_3 = 3.0*Pi/3.0;
const double Pi3_4 = 3.0*Pi/4.0;

#define _Pi 3.141592654
#define _Pi_2 1.570796327
#define _Pi_3 1.047197551
#define _Pi_4 0.785398163
 
#define _Pi2 6.283185307
#define _Pi2_2 _Pi
#define _Pi2_3 2.094395102
#define _Pi2_4 _Pi_2

#define _Pi3 9.424777961
#define _Pi3_2 4.71238898
#define _Pi3_3 _Pi
#define _Pi3_4 2.35619449

class Vector
{
public:
	Vector() : X(0.0), Y(0.0), Z(0.0) { }
	Vector(double x, double y, double z) : X(x), Y(y), Z(z) { }
	
	Vector operator+(const Vector& V) const { return Vector(X+V.X, Y+V.Y, Z+V.Z); }
	Vector operator-(const Vector& V) const { return Vector(X-V.X, Y-V.Y, Z-V.Z); }
	
	const Vector& operator+=(const Vector& V) { X += V.X; Y += V.Y; Z += V.Z; return *this; }
	const Vector& operator-=(const Vector& V) { X -= V.X; Y -= V.Y; Z -= V.Z; return *this; }
	
	// Dot product
	double operator*(const Vector& V) const { return X*V.X + Y*V.Y + Z*V.Z; }
	// Cross product
	Vector operator^(const Vector& V) const { return Vector(Y*V.Z - Z*V.Y, Z*V.X - X*V.Z, X*V.Y - Y*V.X); }

	Vector operator*(double V) const { return Vector(X*V, Y*V, Z*V); }
	Vector operator/(double V) const { if (V == 0.0) V = 1.0; return Vector(X/V, Y/V, Z/V); }
	
	const Vector& operator*=(double V) { X *= V; Y *= V; Z *= V; return *this; }
	const Vector& operator/=(double V) { if (V == 0.0) V = 1.0; X /= V; Y /= V; Z /= V; return *this; }
	
	Vector operator+() const { return *this; }
	Vector operator-() const { return Vector(-X, -Y, -Z); }

	double Magnitude() const { return sqrt(X*X + Y*Y + Z*Z); }
	void Normalise() { double M = Magnitude(); if (M == 0.0) M = 1.0; X /= M; Y /= M; Z /= M; }
	Vector Normalised() const { double M = Magnitude(); if (M == 0.0) M = 1.0; return Vector(X / M, Y / M, Z / M); }
	
	Vector operator()(double x, double y, double z) { X = x; Y = y; Z = z; return *this; }

	double X, Y, Z;
};

inline Vector operator*(double D, const Vector& V)
{
	return V * D;
}

const Vector XX = Vector(1.0, 0.0, 0.0);
const Vector YY = Vector(0.0, 1.0, 0.0);
const Vector ZZ = Vector(0.0, 0.0, 1.0);

// Vector Helpers

// Reflect Point in a plane the passes through the origin with normal Normal.
Vector Reflect(const Vector& Point, const Vector& Normal);

// Project Point perpendicularly onto a plane that passes through the origin with normal Normal.
Vector ProjectOntoPlane(const Vector& Point, const Vector& Normal);

// Shortest distance from Point to a line joining P and Q. Includes end effects.
double DistanceToLine(const Vector& P, const Vector& Q, const Vector& Point);

// Distance between vectors.
double Distance(const Vector& P, const Vector& Q);

// Reflect Point in a plane at position Offset with normal Normal.
Vector ReflectOffset(const Vector& Point, const Vector& Offset, const Vector& Normal);

// Project Point perpendicularly onto an offset plane.
Vector ProjectOntoPlaneOffset(const Vector& Point, const Vector& Offset, const Vector& Normal);

// Project Point in the direction Project onto an offset plane.
Vector Project(const Vector& Point, const Vector& Project, const Vector& Offset, const Vector& Normal);

// Rotate a point about an axis. Offset is a point on the axis, Axis is the axis * the rotation.
Vector Rotate(const Vector& Point, const Vector& Offset, const Vector& Axis);
