#include "ViewWidget.h"

#include <QMouseEvent>

double PsiMax(double alpha)
{
	if (alpha > 0.0)
		return acos(pow(sin(alpha),2)-1);
	return Pi2 - acos(pow(sin(alpha),2)-1);
}

ViewWidget::ViewWidget(QWidget* parent) : QGLWidget(parent)
{
	alpha = 0;
	beta = 0;
	psi = 1;
	
	longBlock.Build(alpha/500.0);
	shortBlock.Build(alpha/500.0, beta/500.0, PsiMax(alpha/500.0) * psi/1000.0);

	viewTheta = 45.0;
	viewPhi = 45.0;
	viewZoom = 300.0;
}
ViewWidget::~ViewWidget()
{ 
	//	makeCurrent();
	//	glDeleteLists(object, 1);
}
QSize ViewWidget::minimumSizeHint() const
{
	return QSize(50, 50);
}

QSize ViewWidget::sizeHint() const
{
	return QSize(500, 500);
}

void normalizeAngle(int& angle)
{
	while (angle < 0)
		angle += 360 * 16;
	while (angle > 360 * 16)
		angle -= 360 * 16;
}

void ViewWidget::setAlpha(int angle)
{
	alpha = angle;
	longBlock.Build(alpha/500.0);
	shortBlock.Build(alpha/500.0, beta/500.0, PsiMax(alpha/500.0) * psi/1000.0);
	updateGL();
}

void ViewWidget::setBeta(int angle)
{
	beta = angle;
	longBlock.Build(alpha/500.0);
	shortBlock.Build(alpha/500.0, beta/500.0, PsiMax(alpha/500.0) * psi/1000.0);
	updateGL();
}

void ViewWidget::setPsi(int angle)
{
	psi = angle;
	longBlock.Build(alpha/500.0);
	shortBlock.Build(alpha/500.0, beta/500.0, PsiMax(alpha/500.0) * psi/1000.0);
	updateGL();
}

void ViewWidget::initializeGL()
{
	// Background colour.
	glClearColor(1.0, 1.0, 1.0, 0.0);
	// Transparency.
	glEnable(GL_BLEND);
	// Depth testing.
	glEnable(GL_DEPTH_TEST);
	// Don't show the back of polys.
	glEnable(GL_CULL_FACE);
	// Smoothing.
	glEnable(GL_POLYGON_SMOOTH);
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	
	// Enable lighting and normal normalization.
	glEnable(GL_LIGHTING);
	glEnable(GL_NORMALIZE);
	
	// Ambient lighting.
	GLfloat ambient[4] = {0.5, 0.5, 0.5, 1.0};
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient);
	
	// Light 0.
	glEnable(GL_LIGHT0);	
	GLfloat diffuse0[] = { 0.8, 0.8, 0.8, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse0);
	// Light 1.
	glEnable(GL_LIGHT1);	
	GLfloat diffuse1[] = { 0.8, 0.8, 0.8, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse1);
	
	// Colouring
	glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);
	
	// Fog
	//	glEnable(GL_FOG); When drawing the grid.
	float  fog[] = {1.0f, 1.0f, 1.0f, 0.0f};
	glFogf(GL_FOG_START,50.0f);
	glFogf(GL_FOG_END, 100.0f);
	glFogfv(GL_FOG_COLOR, fog);      
	glFogi(GL_FOG_MODE, GL_EXP);
	glFogf(GL_FOG_DENSITY, 0.04f);

}

void ViewWidget::paintGL()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	
	glTranslated(0.0, 0.0, -viewZoom/10.0);
	glRotated(viewTheta, -1.0, 0.0, 0.0);
	glRotated(viewPhi, 0.0, 0.0, 1.0);
	
	
	GLfloat pos0[] = { 20.0, -15.0, 10.0, 1.0 }; 
	glLightfv(GL_LIGHT0, GL_POSITION, pos0);
	GLfloat pos1[] = { -5.0, -15.0, 10.0, 1.0 }; 
	glLightfv(GL_LIGHT1, GL_POSITION, pos1);
	
	ShortBlock shortBlock1 = shortBlock.Reflected(shortBlock.GetPlaneOffset(), shortBlock.GetPlaneNormal());
	LongBlock longBlock1 = longBlock.Reflected(shortBlock.GetPlaneOffset(), shortBlock.GetPlaneNormal());
	
	ShortBlock shortBlock2 = shortBlock.Reflected(Vector(), YY);
	LongBlock longBlock2 = longBlock.Reflected(Vector(), YY);
	
	ShortBlock shortBlock3 = shortBlock1.Reflected(Vector(), YY);
	LongBlock longBlock3 = longBlock1.Reflected(Vector(), YY);
	
	Vector C = (longBlock.Centre() + longBlock3.Centre())/2.0;
	glTranslated(-C.X, -C.Y, -C.Z);
	
	glColor4d(0.5, 0.5, 0.5, 1.0);
	shortBlock.Render();
	longBlock.Render();
	shortBlock1.Render();
	longBlock1.Render();
	shortBlock2.Render();
	longBlock2.Render();
	shortBlock3.Render();
	longBlock3.Render();
	
	glLineWidth(3.0f);
	shortBlock.RenderHinges();
	shortBlock3.RenderHinges();
	longBlock1.RenderHinges();
	longBlock2.RenderHinges();
	glLineWidth(1.0f);

	glEnable(GL_FOG);
	glBegin(GL_LINES);
	glColor4d(0.5, 0.5, 0.5, 1.0);
	for (double i = -100.0; i <= 100.0; i += 10.0)
	{
		glVertex3d(-100.0, i, -10.0);
		glVertex3d(100.0, i, -10.0);
		glVertex3d(i, -100.0, -10.0);
		glVertex3d(i, 100.0, -10.0);
	}
	glEnd();
	glDisable(GL_FOG);

}

void ViewWidget::resizeGL(int width, int height)
{
	glViewport(0, 0, width, height);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (width < height)
		glFrustum(-0.1, 0.1, -0.1 * height / width, 0.1 * height / width, 0.2, 200.0);
	else
		glFrustum(-0.1 * width / height, 0.1 * width / height, -0.1, 0.1, 0.2, 200.0);
	glMatrixMode(GL_MODELVIEW);
}

void ViewWidget::mousePressEvent(QMouseEvent *event)
{
	lastPos = event->pos();
}

void ViewWidget::mouseMoveEvent(QMouseEvent *event)
{
	int dx = event->x() - lastPos.x();
	int dy = event->y() - lastPos.y();
	
	if (event->buttons() & Qt::LeftButton || event->buttons() & Qt::MidButton)
	{
		viewPhi += dx;
		viewTheta -= dy;
	}
	if (event->buttons() & Qt::RightButton)
	{
		viewZoom += dy;
	}
	updateGL();
	lastPos = event->pos();
}
void ViewWidget::wheelEvent(QWheelEvent* event)
{
	viewZoom -= event->delta()/2.0;
	updateGL();

}
